"""Cupboard interaction tasks."""
from abc import ABC

import numpy as np

from bigym.bigym_env import BiGymEnv
from bigym.envs.props.cabintets import BaseCabinet, WallCabinet, ModularProp


TOLERANCE = 0.1


class _CupboardsInteractionEnv(BiGymEnv, ABC):
    """Base cupboards environment."""

    _DEFAULT_ROBOT_POS = np.array([-0.2, 0, 1])

    _CUPBOARD_POS = np.array([1.2, 0, 0])
    _CUPBOARD_ROT = np.array([0, 0, -np.pi / 2])
    _CUPBOARD_OFFSET = np.array([0, 0.6, 0])

    def _initialize_env(self):
        self.cupboard_drawers = BaseCabinet(
            self._mojo,
            big_drawers_enable=[True, False],
            small_drawers_enable=[False, False, True, True],
            hob_enable=True,
        )
        self.cupboard_door_left = BaseCabinet(
            self._mojo, door_left_enable=True, shelf_enable=True
        )
        self.cupboard_door_right = BaseCabinet(
            self._mojo, door_right_enable=True, shelf_enable=True
        )
        self.cupboard_wall = WallCabinet(
            self._mojo, doors_enable=True, vent_enable=True
        )

        self.cupboard_drawers.body.set_position(self._CUPBOARD_POS)
        self.cupboard_drawers.body.set_euler(self._CUPBOARD_ROT)

        self.cupboard_door_left.body.set_position(
            self._CUPBOARD_POS - self._CUPBOARD_OFFSET
        )
        self.cupboard_door_left.body.set_euler(self._CUPBOARD_ROT)

        self.cupboard_door_right.body.set_position(
            self._CUPBOARD_POS + self._CUPBOARD_OFFSET
        )
        self.cupboard_door_right.body.set_euler(self._CUPBOARD_ROT)

        self.cupboard_wall.body.set_position(self._CUPBOARD_POS)
        self.cupboard_wall.body.set_euler(self._CUPBOARD_ROT)

    def _get_all_cupboards(self) -> list[ModularProp]:
        return [
            self.cupboard_drawers,
            self.cupboard_door_left,
            self.cupboard_door_right,
            self.cupboard_wall,
        ]


class DrawerTopOpen(_CupboardsInteractionEnv):
    """Open top drawer of the cupboard task."""

    def _success(self) -> bool:
        return np.allclose(self.cupboard_drawers.get_state()[-1], 1, atol=TOLERANCE)


class DrawerTopClose(_CupboardsInteractionEnv):
    """Close top drawer of the cupboard task."""

    def _success(self) -> bool:
        return np.allclose(self.cupboard_drawers.get_state()[-1], 0, atol=TOLERANCE)

    def _on_reset(self):
        self.cupboard_drawers.set_state(np.array([0, 0, 1]))


class DrawersAllOpen(_CupboardsInteractionEnv):
    """Open all drawers of the cupboard task."""

    def _success(self) -> bool:
        return np.allclose(self.cupboard_drawers.get_state(), 1, atol=TOLERANCE)


class DrawersAllClose(_CupboardsInteractionEnv):
    """Close all drawers of the cupboard task."""

    def _success(self) -> bool:
        return np.allclose(self.cupboard_drawers.get_state(), 0, atol=TOLERANCE)

    def _on_reset(self):
        self.cupboard_drawers.set_state(np.array([1, 1, 1]))


class WallCupboardOpen(_CupboardsInteractionEnv):
    """Open doors of the wall cupboard task."""

    def _success(self) -> bool:
        return np.allclose(self.cupboard_wall.get_state(), 1, atol=TOLERANCE)


class WallCupboardClose(_CupboardsInteractionEnv):
    """Close doors of the wall cupboard task."""

    def _success(self) -> bool:
        return np.allclose(self.cupboard_wall.get_state(), 0, atol=TOLERANCE)

    def _on_reset(self):
        self.cupboard_wall.set_state(np.array([1, 1]))


class CupboardsOpenAll(_CupboardsInteractionEnv):
    """Open all doors/drawers of the kitchen counter task."""

    def _success(self) -> bool:
        for cupboard in self._get_all_cupboards():
            if not np.allclose(cupboard.get_state(), 1, atol=TOLERANCE):
                return False
        return True


class CupboardsCloseAll(_CupboardsInteractionEnv):
    """Close all doors/drawers of the kitchen counter task."""

    def _success(self) -> bool:
        for cupboard in self._get_all_cupboards():
            if not np.allclose(cupboard.get_state(), 0, atol=TOLERANCE):
                return False
        return True

    def _on_reset(self):
        for cupboard in self._get_all_cupboards():
            cupboard.set_state(np.ones_like(cupboard.get_state()))
